//Path: T2Editor/js/plugin/video/video.js

class T2VideoPlugin {
    constructor(editor) {
        this.editor = editor;
        this.commands = ['insertYouTube'];
        this.allowedVideoTypes = ['mp4', 'webm', 'ogg'];
    }

    handleCommand(command, button) {
        switch(command) {
            case 'insertYouTube':
                this.insertVideo();
                break;
        }
    }

    onContentSet(html) {
        // 컨텐츠 설정 시 비디오 블록 초기화
        setTimeout(() => {
            this.initializeVideoBlocks();
        }, 100);
    }

    insertVideo() {
        // 현재 커서 위치 저장
        const selection = window.getSelection();
        const range = selection.getRangeAt(0);
        const savedRange = range.cloneRange();

        const modalContent = `
            <div class="t2-video-editor-modal">
                <h3>비디오 삽입</h3>
                <div class="t2-video-tabs">
                    <button class="t2-tab active" data-tab="url">동영상 URL</button>
                    <button class="t2-tab" data-tab="upload">파일 업로드</button>
                </div>
                <div class="t2-tab-content">
                    <div class="t2-tab-pane active" data-pane="url">
                        <input type="text" placeholder="동영상 링크 삽입" class="t2-youtube-url">
                        <div class="t2-video-type-info">
                            지원 동영상 유형: 유튜브, 비디오 파일(.mp4, .webm, .ogg) 링크
                        </div>
                    </div>
                    <div class="t2-tab-pane" data-pane="upload">
                        <div class="t2-video-upload-area">
                            <span class="material-icons">cloud_upload</span>
                            <div class="t2-video-upload-text">클릭하여 동영상 선택</div>
                            <div class="t2-video-upload-hint">지원 형식: MP4, WebM, Ogg (최대 50MB)</div>
                            <input type="file" accept=".mp4,.webm,.ogg,video/mp4,video/webm,video/ogg" />
                        </div>
                        <div class="t2-video-preview-container"></div>
                        <div class="t2-upload-progress" style="display: none;">
                            <div class="t2-progress-bar">
                                <div class="t2-progress-fill"></div>
                            </div>
                            <div class="t2-progress-text">동영상 업로드 중...</div>
                        </div>
                    </div>
                </div>
                <div class="t2-btn-group">
                    <button class="t2-btn" data-action="cancel">취소</button>
                    <button class="t2-btn" data-action="insert">삽입</button>
                </div>
            </div>
        `;

        const modal = T2Utils.createModal(modalContent);
        this.setupVideoModalEvents(modal, savedRange);
    }

    setupVideoModalEvents(modal, savedRange) {
        const urlInput = modal.querySelector('.t2-youtube-url');
        const fileInput = modal.querySelector('input[type="file"]');
        const uploadArea = modal.querySelector('.t2-video-upload-area');
        const previewContainer = modal.querySelector('.t2-video-preview-container');
        const progressContainer = modal.querySelector('.t2-upload-progress');
        const progressBar = modal.querySelector('.t2-progress-fill');
        const progressText = modal.querySelector('.t2-progress-text');
        const insertBtn = modal.querySelector('[data-action="insert"]');
        
        let uploadedVideoUrl = null;
        let activeTab = 'url';

        // 탭 전환 처리
        modal.querySelectorAll('.t2-tab').forEach(tab => {
            tab.addEventListener('click', () => {
                modal.querySelectorAll('.t2-tab').forEach(t => t.classList.remove('active'));
                tab.classList.add('active');
                
                const targetPane = tab.dataset.tab;
                activeTab = targetPane;
                
                modal.querySelectorAll('.t2-tab-pane').forEach(pane => {
                    pane.classList.remove('active');
                    if (pane.dataset.pane === targetPane) {
                        pane.classList.add('active');
                    }
                });

                // 삽입 버튼 상태 업데이트
                if (activeTab === 'url') {
                    insertBtn.disabled = !urlInput.value.trim();
                } else {
                    insertBtn.disabled = !uploadedVideoUrl;
                }
            });
        });

        // URL 입력 이벤트
        urlInput.addEventListener('input', () => {
            insertBtn.disabled = !urlInput.value.trim();
        });

        // 파일 선택 이벤트
        fileInput.addEventListener('change', async (e) => {
            if (e.target.files.length > 0) {
                const result = await this.handleVideoFileUpload(
                    e.target.files[0], 
                    previewContainer, 
                    progressContainer, 
                    progressBar, 
                    progressText, 
                    insertBtn
                );
                if (result) {
                    uploadedVideoUrl = result;
                }
            }
        });

        // 드래그 앤 드롭 처리
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('drag-over');
        });

        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('drag-over');
        });

        uploadArea.addEventListener('drop', async (e) => {
            e.preventDefault();
            uploadArea.classList.remove('drag-over');
            
            if (e.dataTransfer.files.length > 0) {
                const file = e.dataTransfer.files[0];
                if (this.validateVideoFile(file)) {
                    fileInput.files = e.dataTransfer.files;
                    const result = await this.handleVideoFileUpload(
                        file, 
                        previewContainer, 
                        progressContainer, 
                        progressBar, 
                        progressText, 
                        insertBtn
                    );
                    if (result) {
                        uploadedVideoUrl = result;
                    }
                }
            }
        });

        // 동영상 삽입 함수
        const insertVideo = () => {
            let videoInfo = null;
            
            if (activeTab === 'url') {
                const url = urlInput.value.trim();
                videoInfo = T2Utils.getVideoType(url);
                
                if (!videoInfo) {
                    T2Utils.showNotification('올바른 비디오 URL을 입력해주세요.', 'error');
                    return;
                }
            } else if (activeTab === 'upload' && uploadedVideoUrl) {
                videoInfo = { type: 'video', url: uploadedVideoUrl };
            } else {
                T2Utils.showNotification('동영상을 선택해주세요.', 'error');
                return;
            }

            const videoBlock = this.createVideoBlock(videoInfo);
            
            // 저장된 커서 위치에 삽입
            const selection = window.getSelection();
            selection.removeAllRanges();
            selection.addRange(savedRange);
            
            const currentBlock = this.editor.getClosestBlock(savedRange.startContainer);
            if (currentBlock && currentBlock !== this.editor.editor) {
                const topBreak = document.createElement('p');
                topBreak.innerHTML = '<br>';
                currentBlock.parentNode.insertBefore(topBreak, currentBlock.nextSibling);
                
                const wrapper = document.createElement('p');
                wrapper.appendChild(videoBlock);
                topBreak.parentNode.insertBefore(wrapper, topBreak.nextSibling);
                
                const bottomBreak = document.createElement('p');
                bottomBreak.innerHTML = '<br>';
                wrapper.parentNode.insertBefore(bottomBreak, wrapper.nextSibling);
                
                const newRange = document.createRange();
                newRange.setStartAfter(bottomBreak);
                newRange.collapse(true);
                selection.removeAllRanges();
                selection.addRange(newRange);
            }

            this.editor.normalizeContent();
            this.editor.createUndoPoint();
            this.editor.autoSave();
            modal.remove();
        };

        modal.querySelector('[data-action="cancel"]').onclick = () => modal.remove();
        modal.querySelector('[data-action="insert"]').onclick = insertVideo;
        
        urlInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                insertVideo();
            }
        });

        urlInput.focus();
    }

    validateVideoFile(file) {
        const fileExt = file.name.toLowerCase().split('.').pop();
        
        if (!this.allowedVideoTypes.includes(fileExt)) {
            T2Utils.showNotification('지원되지 않는 동영상 형식입니다. MP4, WebM, Ogg 파일만 업로드 가능합니다.', 'error');
            return false;
        }
        
        // 파일 크기 제한 (50MB)
        if (file.size > 50 * 1024 * 1024) {
            T2Utils.showNotification('파일 크기가 너무 큽니다. 최대 50MB까지 업로드 가능합니다.', 'error');
            return false;
        }
        
        return true;
    }

    async handleVideoFileUpload(file, previewContainer, progressContainer, progressBar, progressText, insertBtn) {
        if (!this.validateVideoFile(file)) {
            return null;
        }

        insertBtn.disabled = true;
        progressContainer.style.display = 'block';
        progressBar.style.width = '0%';
        progressText.textContent = '동영상 업로드 중...';

        try {
            // FormData 생성
            const formData = new FormData();
            formData.append('bf_file', file);
            formData.append('uid', this.editor.generateUid());

            // 파일 업로드 (file_upload.php 사용)
            const response = await fetch(`${t2editor_url}/plugin/file/file_upload.php`, {
                method: 'POST',
                body: formData
            });

            const data = await response.json();
            
            if (data.success) {
                progressBar.style.width = '100%';
                progressText.textContent = '업로드 완료';
                
                // 프리뷰 생성
                previewContainer.innerHTML = `
                    <div class="t2-video-preview">
                        <video controls style="width: 100%; max-height: 200px;">
                            <source src="${data.file.url}" type="video/${file.name.split('.').pop()}">
                        </video>
                        <div class="t2-video-info">
                            <span class="t2-video-file-name" title="${file.name}">${file.name}</span>
                            <span class="t2-video-file-size">${T2Utils.formatFileSize(file.size)}</span>
                        </div>
                    </div>
                `;
                
                insertBtn.disabled = false;
                
                // 진행 표시 숨기기
                setTimeout(() => {
                    progressContainer.style.display = 'none';
                }, 1000);
                
                T2Utils.showNotification('동영상이 성공적으로 업로드되었습니다.', 'success');
                
                // 업로드된 비디오 URL 반환
                return data.file.url;
            } else {
                throw new Error(data.message || '업로드 실패');
            }
        } catch (error) {
            console.error('동영상 업로드 오류:', error);
            T2Utils.showNotification('동영상 업로드 중 오류가 발생했습니다.', 'error');
            insertBtn.disabled = false;
            progressContainer.style.display = 'none';
            return null;
        }
    }

    createVideoBlock(videoInfo) {
        const defaultWidth = 320;
        const defaultHeight = 180;

        const wrapper = document.createElement('div');
        wrapper.className = 't2-media-block';
        
        const videoContainer = document.createElement('div');
        videoContainer.style.width = defaultWidth + 'px';
        videoContainer.style.height = defaultHeight + 'px';
        videoContainer.style.maxWidth = '100%';
        videoContainer.style.margin = '0 auto';
        videoContainer.dataset.width = defaultWidth;
        videoContainer.dataset.height = defaultHeight;
        
        let videoElement;
        
        if (videoInfo.type === 'youtube') {
            videoElement = document.createElement('iframe');
            videoElement.src = `https://www.youtube.com/embed/${videoInfo.id}`;
            videoElement.frameBorder = "0";
            videoElement.allow = "accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture";
            videoElement.allowFullscreen = true;
        } else {
            videoElement = document.createElement('video');
            videoElement.src = videoInfo.url;
            videoElement.controls = true;
            videoElement.style.backgroundColor = '#000';
        }
        
        videoElement.style.width = '100%';
        videoElement.style.height = '100%';
        
        videoContainer.appendChild(videoElement);

        const controls = this.createVideoControls(videoContainer, videoElement, videoInfo, defaultWidth, defaultHeight);
        
        wrapper.appendChild(videoContainer);
        wrapper.appendChild(controls);
        
        return wrapper;
    }

    createVideoControls(container, videoElement, videoInfo, defaultWidth, defaultHeight) {
        const controls = document.createElement('div');
        controls.className = 't2-media-controls';
        controls.contentEditable = false;

        const editorWidth = this.editor.editor.clientWidth;
        const maxWidthPercentage = Math.min(100, Math.floor((editorWidth / defaultWidth) * 100));
        const percentage = Math.round((parseInt(container.style.width) / defaultWidth) * 100);

        controls.innerHTML = `
            <button class="t2-btn" onclick="event.preventDefault(); event.stopPropagation(); this.closest('.t2-media-block').remove()">
                <span class="material-icons">delete</span>
            </button>
            <button class="t2-btn edit-url-btn">
                <span class="material-icons">edit</span>
            </button>
            <input type="range" min="30" max="${maxWidthPercentage}" value="${percentage}" style="width: 100px;">
        `;

        // 크기 조절 이벤트
        const rangeInput = controls.querySelector('input[type="range"]');
        if (rangeInput) {
            const resizeObserver = new ResizeObserver(() => {
                const newEditorWidth = this.editor.editor.clientWidth;
                const newMaxPercentage = Math.min(100, Math.floor((newEditorWidth / defaultWidth) * 100));
                rangeInput.max = newMaxPercentage;
                
                if (parseInt(rangeInput.value) > newMaxPercentage) {
                    rangeInput.value = newMaxPercentage;
                    const newWidth = Math.round((defaultWidth * newMaxPercentage) / 100);
                    const newHeight = Math.round((defaultHeight * newMaxPercentage) / 100);
                    
                    container.style.width = `${newWidth}px`;
                    container.style.height = `${newHeight}px`;
                    container.style.maxWidth = '100%';
                    videoElement.style.width = '100%';
                    videoElement.style.height = '100%';
                }
            });
            
            resizeObserver.observe(this.editor.editor);

            rangeInput.addEventListener('input', (e) => {
                const percentage = parseInt(e.target.value);
                const newWidth = Math.round((defaultWidth * percentage) / 100);
                const newHeight = Math.round((defaultHeight * percentage) / 100);
                
                container.style.width = `${newWidth}px`;
                container.style.height = `${newHeight}px`;
                container.style.maxWidth = '100%';
                videoElement.style.width = '100%';
                videoElement.style.height = '100%';
            });
        }

        // URL 편집 이벤트
        const editUrlBtn = controls.querySelector('.edit-url-btn');
        if (editUrlBtn) {
            editUrlBtn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                this.showVideoUrlEditModal(videoElement, videoInfo);
            });
        }

        return controls;
    }

    showVideoUrlEditModal(videoElement, currentVideoInfo) {
        const currentUrl = currentVideoInfo.type === 'youtube' 
            ? `https://youtube.com/watch?v=${currentVideoInfo.id}`
            : currentVideoInfo.url;
        
        const modalContent = `
            <div class="t2-video-editor-modal">
                <h3>비디오 URL 수정</h3>
                <input type="text" placeholder="동영상 링크 삽입" class="t2-youtube-url" value="${currentUrl}">
                <div class="t2-video-type-info">
                    지원 동영상 유형: 유튜브, 비디오 파일(.mp4, .webm, .ogg) 링크
                </div>
                <div class="t2-btn-group">
                    <button class="t2-btn" data-action="cancel">취소</button>
                    <button class="t2-btn" data-action="insert">수정</button>
                </div>
            </div>
        `;

        const modal = T2Utils.createModal(modalContent);
        
        const updateVideo = () => {
            const url = modal.querySelector('.t2-youtube-url').value;
            const videoInfo = T2Utils.getVideoType(url);
            
            if (!videoInfo) {
                T2Utils.showNotification('올바른 비디오 URL을 입력해주세요.', 'error');
                return;
            }

            if (videoInfo.type === 'youtube') {
                videoElement.src = `https://www.youtube.com/embed/${videoInfo.id}`;
            } else {
                const newVideo = document.createElement('video');
                newVideo.src = videoInfo.url;
                newVideo.controls = true;
                newVideo.style = videoElement.style;
                videoElement.parentNode.replaceChild(newVideo, videoElement);
            }

            modal.remove();
            this.editor.createUndoPoint();
            this.editor.autoSave();
        };

        modal.querySelector('[data-action="cancel"]').onclick = () => modal.remove();
        modal.querySelector('[data-action="insert"]').onclick = updateVideo;
        
        modal.querySelector('.t2-youtube-url').addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                updateVideo();
            }
        });

        modal.querySelector('.t2-youtube-url').focus();
    }

    initializeVideoBlocks() {
        // 유튜브 iframe 초기화
        this.editor.editor.querySelectorAll('iframe[src*="youtube"]:not(.t2-media-block iframe)').forEach(frame => {
            const width = parseInt(frame.style.width) || 320;
            const height = parseInt(frame.style.height) || 180;
            
            const videoId = frame.src.match(/embed\/([^?]+)/)?.[1];
            if (videoId) {
                const wrapper = this.createVideoBlock({ type: 'youtube', id: videoId });
                frame.parentNode.replaceChild(wrapper, frame);
            }
        });

        // 비디오 요소 초기화
        this.editor.editor.querySelectorAll('video:not(.t2-media-block video)').forEach(video => {
            const width = parseInt(video.style.width) || 320;
            const height = parseInt(video.style.height) || 180;
            
            const wrapper = this.createVideoBlock({ type: 'video', url: video.src });
            video.parentNode.replaceChild(wrapper, video);
        });

        // 기존 미디어 블록 처리
        this.editor.editor.querySelectorAll('.t2-media-block').forEach(block => {
            const container = block.querySelector('div:first-child');
            const mediaElement = container?.querySelector('iframe, video');
            
            if (mediaElement && (mediaElement.tagName === 'IFRAME' || mediaElement.tagName === 'VIDEO')) {
                const currentWidth = parseInt(container.style.width) || 320;
                const currentHeight = parseInt(container.style.height) || 180;
                
                if (!container.style.maxWidth) {
                    container.style.maxWidth = '100%';
                }
                if (!container.style.margin) {
                    container.style.margin = '0 auto';
                }
                
                mediaElement.style.width = '100%';
                mediaElement.style.height = '100%';
                
                // 블록이 p 태그 안에 있도록 처리
                if (block.parentNode.nodeName !== 'P') {
                    const p = document.createElement('p');
                    block.parentNode.insertBefore(p, block);
                    p.appendChild(block);
                }
                
                // 기존 컨트롤 제거
                const existingControls = block.querySelector('.t2-media-controls');
                if (existingControls) {
                    existingControls.remove();
                }
                
                let videoInfo;
                if (mediaElement.tagName === 'IFRAME') {
                    const videoId = mediaElement.src.match(/embed\/([^?]+)/)?.[1];
                    videoInfo = { type: 'youtube', id: videoId };
                } else {
                    videoInfo = { type: 'video', url: mediaElement.src };
                }
                
                const controls = this.createVideoControls(container, mediaElement, videoInfo, currentWidth, currentHeight);
                block.appendChild(controls);
            }
        });
    }
}

// 스타일 추가 (CSS)
const style = document.createElement('style');
style.textContent = `
    .t2-video-tabs {
        display: flex;
        gap: 10px;
        margin-bottom: 20px;
        border-bottom: 1px solid #e5e7eb;
        padding-bottom: 10px;
    }
    
    .t2-video-upload-area {
        border: 2px dashed #e5e7eb;
        border-radius: 8px;
        padding: 32px 16px;
        text-align: center;
        cursor: pointer;
        transition: all 0.2s;
        margin-bottom: 20px;
        position: relative;
    }
    
    .t2-video-upload-area:hover,
    .t2-video-upload-area.drag-over {
        border-color: #2563eb;
        background: rgba(37,99,235,.05);
    }
    
    .t2-video-upload-area .material-icons {
        font-size: 48px;
        color: #6b7280;
        margin-bottom: 12px;
    }
    
    .t2-video-upload-text {
        font-size: 15px;
        color: #4b5563;
        margin-bottom: 4px;
    }
    
    .t2-video-upload-hint {
        font-size: 13px;
        color: #6b7280;
    }
    
    .t2-video-upload-area input[type="file"] {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        opacity: 0;
        cursor: pointer;
    }
    
    .t2-video-preview-container {
        margin-bottom: 20px;
    }
    
    .t2-video-preview {
        background: #f9fafb;
        border-radius: 8px;
        padding: 10px;
    }
    
    .t2-video-preview video {
        border-radius: 6px;
        margin-bottom: 10px;
    }
    
    .t2-video-info {
        display: flex;
        justify-content: space-between;
        font-size: 13px;
        color: #6b7280;
        gap: 10px;
    }
    
    /* 파일명 오버플로우 처리 */
    .t2-video-file-name {
        flex: 1;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
        max-width: 70%;
    }
    
    .t2-video-file-size {
        flex-shrink: 0;
    }
    
    /* 다크모드 지원 */
    html[data-t2editor-theme="dark"] .t2-video-upload-area {
        border-color: #444;
        color: #e0e0e0;
    }
    
    html[data-t2editor-theme="dark"] .t2-video-upload-area:hover,
    html[data-t2editor-theme="dark"] .t2-video-upload-area.drag-over {
        border-color: #0187fe;
        background-color: rgba(1,135,254,.1);
    }
    
    html[data-t2editor-theme="dark"] .t2-video-upload-area .material-icons {
        color: #aaa;
    }
    
    html[data-t2editor-theme="dark"] .t2-video-upload-text {
        color: #e0e0e0;
    }
    
    html[data-t2editor-theme="dark"] .t2-video-upload-hint {
        color: #aaa;
    }
    
    html[data-t2editor-theme="dark"] .t2-video-preview {
        background: #383838;
    }
    
    html[data-t2editor-theme="dark"] .t2-video-info {
        color: #aaa;
    }
`;

// 스타일을 문서에 추가 (플러그인 로드 시)
if (!document.querySelector('#t2-video-plugin-styles')) {
    style.id = 't2-video-plugin-styles';
    document.head.appendChild(style);
}

window.T2VideoPlugin = T2VideoPlugin;